<?php
header('Content-Type: application/json');
include('../db/dbi.php');

// --------------------------------------
// ENHANCED LOGGER
// --------------------------------------
function logTrade($data, $raw = '', $broker = 'Unknown') {
    $log = "=== NEW TRADE " . date("Y-m-d H:i:s") . " ===\n";
    $log .= "Broker    : " . $broker . "\n";
    $log .= "Account   : " . (isset($data['acct_number']) ? $data['acct_number'] : 'N/A') . "\n";
    $log .= "Ticket    : " . (isset($data['ticket']) ? $data['ticket'] : 'N/A') . "\n";
    $log .= "Symbol    : " . (isset($data['symbol']) ? $data['symbol'] : 'N/A') . "\n";
    $log .= "Type      : " . (isset($data['type']) ? $data['type'] : 'N/A') . "\n";
    $log .= "Status    : " . (isset($data['status']) ? $data['status'] : 'N/A') . "\n";
    $log .= "OpenTime  : " . (isset($data['open_time']) ? $data['open_time'] : 'N/A') . "\n";
    $log .= "CloseTime : " . (isset($data['close_time']) ? $data['close_time'] : 'NULL') . "\n";
    $log .= "Comment   : " . (isset($data['comment']) ? $data['comment'] : '') . "\n";
    $log .= "Magic     : " . (isset($data['magic_number']) ? $data['magic_number'] : 'N/A') . "\n";
    
    // DEBUG: Log all data keys and comment fields
    $log .= "Data keys : " . implode(', ', array_keys($data)) . "\n";
    $log .= "Has 'comment' field: " . (isset($data['comment']) ? 'YES' : 'NO') . "\n";
    $log .= "Has 'comments' field: " . (isset($data['comments']) ? 'YES' : 'NO') . "\n";
    $log .= "Comment value: " . (isset($data['comment']) ? $data['comment'] : (isset($data['comments']) ? $data['comments'] : 'EMPTY')) . "\n";
    
    $log .= "RAW JSON  :\n" . $raw . "\n";
    $log .= str_repeat("=", 70) . "\n\n";
    //file_put_contents('trades_received.log', $log, FILE_APPEND | LOCK_EX);
}

// --------------------------------------
// ENHANCED BROKER DETECTION
// --------------------------------------
function detectBroker($data) {
    // Check for broker-specific identifiers
    if (isset($data['broker'])) {
        return strtoupper(trim($data['broker']));
    }
    
    // Try to detect from account number patterns
    $acct = 0;
    if (isset($data['acct_number'])) $acct = (int)$data['acct_number'];
    elseif (isset($data['account'])) $acct = (int)$data['account'];
    elseif (isset($data['Account'])) $acct = (int)$data['Account'];
    
    $comment = '';
    if (isset($data['comment'])) $comment = trim($data['comment']);
    elseif (isset($data['comments'])) $comment = trim($data['comments']);
    elseif (isset($data['Comment'])) $comment = trim($data['Comment']);
    
    // Check MT5/MT4 account patterns
    if ($acct > 0) {
        // FBS typically uses certain number ranges
        if ($acct >= 70000000 && $acct <= 79999999) {
            return 'FBS';
        }
        // Exness patterns
        if ($acct >= 80000000 && $acct <= 89999999) {
            return 'EXNESS';
        }
        // ICMarkets patterns
        if ($acct >= 10000000 && $acct <= 19999999) {
            return 'ICMARKETS';
        }
        // XM patterns
        if ($acct >= 20000000 && $acct <= 29999999) {
            return 'XM';
        }
    }
    
    // Check comment field for broker names
    $commentLower = strtolower($comment);
    $brokerKeywords = [
        'fbs' => 'FBS',
        'exness' => 'EXNESS',
        'ic markets' => 'ICMARKETS',
        'icmarkets' => 'ICMARKETS',
        'xm.com' => 'XM',
        'xm-' => 'XM',
        'forex.com' => 'FOREXCOM',
        'oanda' => 'OANDA',
        'pepperstone' => 'PEPPERSTONE',
        'tickmill' => 'TICKMILL',
        'hugosway' => 'HUGOSWAY',
        'tradersway' => 'TRADERSWAY',
        'fxpro' => 'FXPRO'
    ];
    
    foreach ($brokerKeywords as $keyword => $brokerName) {
        if (strpos($commentLower, $keyword) !== false) {
            return $brokerName;
        }
    }
    
    // Check for server or other identifiers in the data
    if (isset($data['server'])) {
        $server = strtolower($data['server']);
        $serverKeywords = [
            'fbs' => 'FBS',
            'exness' => 'EXNESS',
            'icmarkets' => 'ICMARKETS',
            'xm' => 'XM',
            'fxpro' => 'FXPRO',
            'pepperstone' => 'PEPPERSTONE'
        ];
        
        foreach ($serverKeywords as $keyword => $brokerName) {
            if (strpos($server, $keyword) !== false) {
                return $brokerName;
            }
        }
    }
    
    // Check for specific symbol patterns
    if (isset($data['symbol'])) {
        $symbol = strtolower($data['symbol']);
        // Some brokers append broker names to symbols
        if (strpos($symbol, '.fbs') !== false) return 'FBS';
        if (strpos($symbol, '.ex') !== false) return 'EXNESS';
        if (strpos($symbol, '.icm') !== false) return 'ICMARKETS';
        if (strpos($symbol, '.xm') !== false) return 'XM';
    }
    
    // Check magic number patterns (some EAs encode broker in magic)
    if (isset($data['magic_number'])) {
        $magic = (int)$data['magic_number'];
        if ($magic >= 100000 && $magic <= 199999) return 'FBS';
        if ($magic >= 200000 && $magic <= 299999) return 'EXNESS';
        if ($magic >= 300000 && $magic <= 399999) return 'ICMARKETS';
    }
    
    return 'UNKNOWN';
}

// --------------------------------------
// NORMALIZE DATETIME
// --------------------------------------
function normalizeDateTime($value, $fieldName = 'datetime') {
    if (empty($value) || $value === 'null' || $value === null || $value === 'NULL') {
        return null;
    }
    
    // Convert various formats to standard MySQL datetime
    $value = trim($value);
    
    // Replace dots with dashes (MT5 format: 2024.12.17 -> 2024-12-17)
    $value = str_replace('.', '-', $value);
    
    // Check for invalid dates
    $invalidDates = [
        '1970-01-01 00:00:00',
        '0000-00-00 00:00:00',
        '1970-01-01',
        '0000-00-00',
        '1899-12-30'
    ];
    
    if (in_array($value, $invalidDates)) {
        return null;
    }
    
    // Try to parse the datetime
    $formats = [
        'Y-m-d H:i:s',
        'Y-m-d H:i',
        'Y-m-d',
        'd-m-Y H:i:s',
        'd-m-Y H:i',
        'd-m-Y',
        'Y.m.d H:i:s',
        'Y.m.d H:i',
        'Y.m.d',
        'd.m.Y H:i:s',
        'd.m.Y H:i',
        'd.m.Y'
    ];
    
    foreach ($formats as $format) {
        $dt = DateTime::createFromFormat($format, $value);
        if ($dt !== false) {
            return $dt->format('Y-m-d H:i:s');
        }
    }
    
    // Try to parse timestamp
    if (is_numeric($value) && strlen($value) >= 10) {
        $timestamp = (int)$value;
        // Handle milliseconds (13 digits) or seconds (10 digits)
        if (strlen($value) == 13) {
            $timestamp = (int)($timestamp / 1000); // Convert milliseconds to seconds
        }
        $dt = new DateTime("@$timestamp");
        return $dt->format('Y-m-d H:i:s');
    }
    
    // If all parsing fails, log and return null
    error_log("Failed to parse $fieldName: $value");
    return null;
}

// --------------------------------------
// NORMALIZE COMMENT FIELD
// --------------------------------------
function normalizeComment($comment, $broker = 'UNKNOWN') {
    if (!isset($comment) || $comment === null || $comment === '') {
        return '';
    }
    
    // Convert to string and trim
    $comment = trim((string)$comment);
    
    // Remove null bytes and other problematic characters
    $comment = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $comment);
    
    // Handle different encodings
    if (!mb_check_encoding($comment, 'UTF-8')) {
        $comment = mb_convert_encoding($comment, 'UTF-8', 'auto');
    }
    
    // Broker-specific comment handling
    switch ($broker) {
        case 'EXNESS':
            // Exness may send empty strings, whitespace, or encoded data
            if ($comment === '' || ctype_space($comment)) {
                return '';
            }
            // Check if it's base64 encoded
            if (base64_encode(base64_decode($comment, true)) === $comment) {
                $decoded = base64_decode($comment);
                if (mb_check_encoding($decoded, 'UTF-8')) {
                    $comment = trim($decoded);
                }
            }
            break;
            
        case 'FBS':
            // FBS usually sends clean comment data
            // Remove any trailing broker-specific codes
            $comment = preg_replace('/\s*FBS-\d+/', '', $comment);
            break;
            
        case 'ICMARKETS':
            // ICM often includes server codes
            $comment = preg_replace('/\s*IC\d+/', '', $comment);
            break;
            
        case 'XM':
            // XM often includes account identifiers
            $comment = preg_replace('/\s*XM-\d+/', '', $comment);
            break;
    }
    
    // Truncate to field length (255 chars for VARCHAR, more for TEXT)
    if (mb_strlen($comment) > 255) {
        $comment = mb_substr($comment, 0, 252) . '...';
    }
    
    return $comment;
}

// --------------------------------------
// EXTRACT AND NORMALIZE DATA
// --------------------------------------
function extractTradeData($data) {
    $broker = detectBroker($data);
    
    // Try multiple possible field names for each field
    $extracted = [
        'broker' => $broker,
        
        // Account number - try multiple field names
        'acct_number' => isset($data['acct_number']) ? (int)$data['acct_number'] : 
                        (isset($data['account']) ? (int)$data['account'] : 
                        (isset($data['Account']) ? (int)$data['Account'] : 
                        (isset($data['account_number']) ? (int)$data['account_number'] : 
                        (isset($data['login']) ? (int)$data['login'] : 0)))),
        
        // Ticket/Order - try multiple field names
        'ticket' => isset($data['ticket']) ? (int)$data['ticket'] : 
                   (isset($data['order']) ? (int)$data['order'] : 
                   (isset($data['Order']) ? (int)$data['Order'] : 
                   (isset($data['Ticket']) ? (int)$data['Ticket'] : 
                   (isset($data['position']) ? (int)$data['position'] : 
                   (isset($data['Position']) ? (int)$data['Position'] : 0))))),
        
        // Symbol - try multiple field names
        'symbol' => isset($data['symbol']) ? strtoupper(trim($data['symbol'])) : 
                   (isset($data['Symbol']) ? strtoupper(trim($data['Symbol'])) : 
                   (isset($data['instrument']) ? strtoupper(trim($data['instrument'])) : 
                   (isset($data['pair']) ? strtoupper(trim($data['pair'])) : ''))),
        
        // Type - try multiple field names
        'type' => isset($data['type']) ? strtoupper(trim($data['type'])) : 
                 (isset($data['Type']) ? strtoupper(trim($data['Type'])) : 
                 (isset($data['operation']) ? strtoupper(trim($data['operation'])) : 
                 (isset($data['cmd']) ? strtoupper(trim($data['cmd'])) : ''))),
        
        // Lot/Volume - try multiple field names
        'lot' => isset($data['lot']) ? (float)$data['lot'] : 
                (isset($data['Lot']) ? (float)$data['Lot'] : 
                (isset($data['volume']) ? (float)$data['volume'] : 
                (isset($data['Volume']) ? (float)$data['Volume'] : 
                (isset($data['size']) ? (float)$data['size'] : 0)))),
        
        // Open price - try multiple field names
        'open_price' => isset($data['open_price']) ? (float)$data['open_price'] : 
                       (isset($data['Open_price']) ? (float)$data['Open_price'] : 
                       (isset($data['price_open']) ? (float)$data['price_open'] : 
                       (isset($data['openPrice']) ? (float)$data['openPrice'] : 
                       (isset($data['entry']) ? (float)$data['entry'] : 0)))),
        
        // Close price - try multiple field names
        'close_price' => isset($data['close_price']) ? (float)$data['close_price'] : 
                        (isset($data['Close_price']) ? (float)$data['Close_price'] : 
                        (isset($data['price_close']) ? (float)$data['price_close'] : 
                        (isset($data['closePrice']) ? (float)$data['closePrice'] : 
                        (isset($data['exit']) ? (float)$data['exit'] : 0)))),
        
        // Profit - try multiple field names
        'profit' => isset($data['profit']) ? (float)$data['profit'] : 
                   (isset($data['Profit']) ? (float)$data['Profit'] : 
                   (isset($data['pl']) ? (float)$data['pl'] : 
                   (isset($data['PL']) ? (float)$data['PL'] : 
                   (isset($data['p/l']) ? (float)$data['p/l'] : 0)))),
        
        // Swap - try multiple field names
        'swap' => isset($data['swap']) ? (float)$data['swap'] : 
                 (isset($data['Swap']) ? (float)$data['Swap'] : 
                 (isset($data['storage']) ? (float)$data['storage'] : 
                 (isset($data['interest']) ? (float)$data['interest'] : 0))),
        
        // Commission - try multiple field names
        'commission' => isset($data['commission']) ? (float)$data['commission'] : 
                      (isset($data['Commission']) ? (float)$data['Commission'] : 
                      (isset($data['fee']) ? (float)$data['fee'] : 
                      (isset($data['Fee']) ? (float)$data['Fee'] : 0))),
        
        // Magic number - try multiple field names
        'magic_number' => isset($data['magic_number']) ? (int)$data['magic_number'] : 
                         (isset($data['Magic_number']) ? (int)$data['Magic_number'] : 
                         (isset($data['magic']) ? (int)$data['magic'] : 
                         (isset($data['Magic']) ? (int)$data['Magic'] : 
                         (isset($data['expert']) ? (int)$data['expert'] : 
                         (isset($data['Expert']) ? (int)$data['Expert'] : 0))))),
        
        // Status - try multiple field names
        'status' => isset($data['status']) ? strtoupper(trim($data['status'])) : 
                   (isset($data['Status']) ? strtoupper(trim($data['Status'])) : 
                   (isset($data['state']) ? strtoupper(trim($data['state'])) : 'OPEN')),
    ];
    
    // Try multiple possible comment field names
    $comment = '';
    $possibleCommentFields = [
        'comment', 'comments', 'Comment', 'Comments', 
        'reason', 'Reason', 'notes', 'Notes',
        'description', 'Description', 'text', 'Text',
        'info', 'Info', 'message', 'Message'
    ];
    
    foreach ($possibleCommentFields as $field) {
        if (isset($data[$field]) && !empty(trim($data[$field]))) {
            $comment = $data[$field];
            error_log("Found comment in field: $field = " . substr($comment, 0, 50));
            break;
        }
    }
    
    // Handle comment with broker-specific normalization
    $extracted['comment'] = normalizeComment($comment, $broker);
    
    // Handle datetime fields - try multiple field names
    $openTimeValue = null;
    $closeTimeValue = null;
    
    $openTimeFields = ['open_time', 'Open_time', 'time_open', 'Time_open', 'openTime', 'opened'];
    $closeTimeFields = ['close_time', 'Close_time', 'time_close', 'Time_close', 'closeTime', 'closed'];
    
    foreach ($openTimeFields as $field) {
        if (isset($data[$field]) && !empty(trim($data[$field]))) {
            $openTimeValue = $data[$field];
            break;
        }
    }
    
    foreach ($closeTimeFields as $field) {
        if (isset($data[$field]) && !empty(trim($data[$field]))) {
            $closeTimeValue = $data[$field];
            break;
        }
    }
    
    $extracted['open_time'] = normalizeDateTime($openTimeValue, 'open_time');
    $extracted['close_time'] = normalizeDateTime($closeTimeValue, 'close_time');
    
    return $extracted;
}

// --------------------------------------
// READ RAW INPUT
// --------------------------------------
$raw = file_get_contents("php://input");
if (!$raw) {
    echo json_encode(["status"=>"error","message"=>"Empty request body"]);
    exit;
}

// Remove BOM and hidden characters
$raw = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/', '', $raw);
$raw = trim($raw, " \t\n\r\0\x0B\xEF\xBB\xBF");

// Decode JSON
$data = json_decode($raw, true);

// Check for JSON errors
if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode([
        "status" => "error",
        "message" => "JSON decode failed: " . json_last_error_msg(),
        "raw" => substr($raw, 0, 200)
    ]);
    exit;
}

// --------------------------------------
// PING SUPPORT
// --------------------------------------
if (isset($data['ping']) && $data['ping'] === true) {
    echo json_encode([
        "status" => "ok",
        "server" => "Royals API Alive",
        "time" => date("Y-m-d H:i:s"),
        "version" => "2.1"
    ]);
    exit;
}

// --------------------------------------
// EXTRACT AND NORMALIZE DATA
// --------------------------------------
$tradeData = extractTradeData($data);

// Log everything
logTrade($data, "RAW CLEANED:\n" . $raw, $tradeData['broker']);

// --------------------------------------
// VALIDATION
// --------------------------------------
if ($tradeData['acct_number'] === 0 || $tradeData['ticket'] === 0 || $tradeData['symbol'] === '') {
    echo json_encode([
        "status" => "error",
        "message" => "Missing required fields",
        "received" => [
            "broker" => $tradeData['broker'],
            "acct" => $tradeData['acct_number'],
            "ticket" => $tradeData['ticket'],
            "symbol" => $tradeData['symbol'],
            "type" => $tradeData['type'],
            "lot" => $tradeData['lot'],
            "status" => $tradeData['status']
        ],
        "data_keys" => array_keys($data)
    ]);
    exit;
}

// --------------------------------------
// ENSURE TABLE STRUCTURE
// --------------------------------------
// First create the table if it doesn't exist
$conn->query("
    CREATE TABLE IF NOT EXISTS trade_history (
        id INT AUTO_INCREMENT PRIMARY KEY,
        broker VARCHAR(50) DEFAULT 'UNKNOWN',
        acct_number INT NOT NULL,
        ticket BIGINT NOT NULL,
        symbol VARCHAR(20) NOT NULL,
        type VARCHAR(10) NOT NULL,
        lot DOUBLE NOT NULL,
        open_price DOUBLE NOT NULL,
        close_price DOUBLE DEFAULT 0,
        profit DOUBLE DEFAULT 0,
        swap DOUBLE DEFAULT 0,
        commission DOUBLE DEFAULT 0,
        comment TEXT,
        magic_number INT DEFAULT 0,
        open_time DATETIME,
        close_time DATETIME DEFAULT NULL,
        status VARCHAR(20) DEFAULT 'OPEN',
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY uq_ticket (ticket),
        INDEX idx_acct (acct_number),
        INDEX idx_status (status),
        INDEX idx_broker (broker)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
");

// Check if broker column exists and add it if not
$result = $conn->query("SHOW COLUMNS FROM trade_history LIKE 'broker'");
if ($result && $result->num_rows === 0) {
    // Add broker column if it doesn't exist
    $conn->query("ALTER TABLE trade_history ADD COLUMN broker VARCHAR(50) DEFAULT 'UNKNOWN' AFTER id");
    $conn->query("ALTER TABLE trade_history ADD INDEX idx_broker (broker)");
}
if ($result) $result->close();

// --------------------------------------
// PREPARE UPSERT
// --------------------------------------
$sql = "
INSERT INTO trade_history
(broker, acct_number, ticket, symbol, type, lot, open_price, close_price, profit, swap, commission, comment, magic_number, open_time, close_time, status)
VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
ON DUPLICATE KEY UPDATE
    broker = VALUES(broker),
    close_price = VALUES(close_price),
    profit = VALUES(profit),
    swap = VALUES(swap),
    commission = VALUES(commission),
    comment = VALUES(comment),
    close_time = VALUES(close_time),
    status = VALUES(status),
    updated_at = CURRENT_TIMESTAMP
";

$stmt = $conn->prepare($sql);
if (!$stmt) {
    echo json_encode(["status"=>"error","message"=>"Prepare failed: ".$conn->error]);
    exit;
}

// --------------------------------------
// BIND PARAMETERS
// --------------------------------------
$stmt->bind_param(
    "siissddddddsssss",
    $tradeData['broker'],
    $tradeData['acct_number'],
    $tradeData['ticket'],
    $tradeData['symbol'],
    $tradeData['type'],
    $tradeData['lot'],
    $tradeData['open_price'],
    $tradeData['close_price'],
    $tradeData['profit'],
    $tradeData['swap'],
    $tradeData['commission'],
    $tradeData['comment'],
    $tradeData['magic_number'],
    $tradeData['open_time'],
    $tradeData['close_time'],
    $tradeData['status']
);

// --------------------------------------
// EXECUTE
// --------------------------------------
if ($stmt->execute()) {
    $action = ($stmt->affected_rows > 1) ? "updated" : "inserted";
    $response = [
        "status" => "success",
        "message" => "Trade $action",
        "broker" => $tradeData['broker'],
        "ticket" => $tradeData['ticket'],
        "account" => $tradeData['acct_number'],
        "symbol" => $tradeData['symbol'],
        "type" => $tradeData['type'],
        "action" => $action,
        "affected_rows" => $stmt->affected_rows,
        "comment_received" => !empty($tradeData['comment']),
        "comment_length" => strlen($tradeData['comment'])
    ];
    echo json_encode($response);
} else {
    echo json_encode([
        "status" => "error",
        "message" => "Execute failed: " . $stmt->error,
        "sql_state" => $stmt->sqlstate,
        "query" => $sql
    ]);
}

$stmt->close();
$conn->close();
?>