<?php
// app/debug-logger.php

class DebugLogger {
    private static $log_file = '../logs/debug.log';
    private static $error_file = '../logs/error.log';
    
    public static function log($message, $data = null) {
        self::writeLog('DEBUG', $message, $data, self::$log_file);
    }
    
    public static function error($message, $data = null) {
        self::writeLog('ERROR', $message, $data, self::$error_file);
    }
    
    public static function logRequest() {
        $request_data = array(
            'timestamp' => date('Y-m-d H:i:s'),
            'method' => $_SERVER['REQUEST_METHOD'],
            'url' => $_SERVER['REQUEST_URI'],
            'get_params' => $_GET,
            'post_params' => $_POST,
            'session_data' => isset($_SESSION) ? $_SESSION : array(),
            'ip' => $_SERVER['REMOTE_ADDR'],
            'user_agent' => $_SERVER['HTTP_USER_AGENT']
        );
        
        self::log('Request received', $request_data);
    }
    
    public static function logLotteryAction($action, $data = null) {
        $log_data = array(
            'action' => $action,
            'data' => $data,
            'user_id' => isset($_SESSION['uid']) ? $_SESSION['uid'] : 'unknown',
            'logid' => isset($_SESSION['logid']) ? $_SESSION['logid'] : 'unknown'
        );
        
        self::log('Lottery Action: ' . $action, $log_data);
    }
    
    private static function writeLog($type, $message, $data, $file) {
        // Create logs directory if it doesn't exist
        $log_dir = dirname(dirname(__FILE__)) . '/logs/';
        if (!is_dir($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        
        $log_entry = "[" . date('Y-m-d H:i:s') . "] [$type] $message\n";
        
        if ($data !== null) {
            $log_entry .= "Data: " . print_r($data, true) . "\n";
        }
        
        // Add stack trace for errors
        if ($type == 'ERROR') {
            $trace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 5);
            $log_entry .= "Stack trace:\n";
            foreach ($trace as $index => $call) {
                if (isset($call['file']) && isset($call['line'])) {
                    $log_entry .= "  #$index {$call['file']}({$call['line']})\n";
                }
            }
        }
        
        $log_entry .= str_repeat("-", 80) . "\n\n";
        
        // Write to file
        file_put_contents($log_dir . basename($file), $log_entry, FILE_APPEND | LOCK_EX);
        
        // Also output to browser if debug mode is on
        if (isset($_GET['debug']) && $_GET['debug'] == 'true') {
            echo "<pre style='background: #f0f0f0; padding: 10px; margin: 10px; border: 1px solid #ccc;'>";
            echo htmlspecialchars($log_entry);
            echo "</pre>";
        }
    }
    
    public static function checkSystem() {
        $checks = array();
        
        // Check PHP version
        $checks['php_version'] = PHP_VERSION;
        
        // Check required extensions
        $required_extensions = array('mysqli', 'json', 'session');
        foreach ($required_extensions as $ext) {
            $checks['ext_' . $ext] = extension_loaded($ext) ? 'OK' : 'MISSING';
        }
        
        // Check file permissions
        $files_to_check = array(
            '../db/db.php' => 'Database connection',
            '../app/functions.php' => 'Functions file',
            '../app/function_1.php' => 'Function 1 file',
            '../app/lottery-helpers.php' => 'Lottery helpers'
        );
        
        foreach ($files_to_check as $file => $desc) {
            $checks['file_' . basename($file)] = file_exists($file) ? 'EXISTS' : 'MISSING';
            if (file_exists($file)) {
                $checks['readable_' . basename($file)] = is_readable($file) ? 'READABLE' : 'NOT READABLE';
            }
        }
        
        // Check session
        $checks['session_status'] = session_status();
        $checks['session_id'] = session_id();
        
        // Log the system check
        self::log('System check performed', $checks);
        
        return $checks;
    }
}